/**
 * カスタムPCパーツ 見積作成 Web UI アプリケーション
 * * 目的: お好みのパソコンパーツをそろえて構成し、見積書を作成すること
 * * 価格に関する注意:
 * - 表示価格は、データインポート時の静的価格です。
 * - 実際の販売価格は購入リンク先（Amazonなど）で必ずご確認ください。
 * - 本ツールは価格変動に関する責任を一切負いません。
 * * （CPUクーラー、アフィリエイト連携、見積出力修正対応済み）
 */

class PCBuilder {
    // 9つのカテゴリを定義 (CPUクーラーを追加)
    CATEGORIES = ['case', 'os', 'cpu', 'cpucooler', 'mb', 'memory', 'vga', 'psu', 'ssd'];

    // プレビューパネルの表示順 (CPUクーラーを追加)
    PREVIEW_ORDER = [
        { key: 'case', name: 'PCケース', icon: 'fas fa-computer' },
        { key: 'os', name: 'OS', icon: 'fas fa-laptop-code' },
        { key: 'cpu', name: 'CPU', icon: 'fas fa-microchip' },
        { key: 'cpucooler', name: 'CPUクーラー', icon: 'fas fa-fan' }, // 新しい項目
        { key: 'mb', name: 'マザーボード', icon: 'fas fa-grip' },
        { key: 'memory', name: 'メモリ', icon: 'fas fa-memory' },
        { key: 'vga', name: 'グラフィックボード', icon: 'fas fa-temperature-arrow-up' },
        { key: 'psu', name: '電源', icon: 'fas fa-bolt' },
        { key: 'ssd', name: 'SSD', icon: 'fas fa-hdd' }
    ];

    // デフォルト製品データ (アフィリエイトURLはダミー)
    DEFAULT_DATA = {
        case: [
            { id: 'case1', name: 'NZXT H5 Flow', price: 15000, specs: 'ミドルタワー, ATX', image: 'https://picsum.photos/400/300?random=1', affiliateUrl: 'https://affiliate-dummy-link.com/case1' },
            { id: 'case2', name: 'Fractal Design Meshify 2', price: 22000, specs: 'E-ATX対応, 高エアフロー', image: 'https://picsum.photos/400/300?random=2', affiliateUrl: '' },
        ],
        os: [
            { id: 'os1', name: 'Windows 11 Pro', price: 25000, specs: 'プロフェッショナル版', image: '', affiliateUrl: 'https://affiliate-dummy-link.com/os1' },
            { id: 'os2', name: 'Windows 11 Home', price: 18000, specs: '一般家庭向け', image: '', affiliateUrl: '' },
        ],
        cpu: [
            { id: 'cpu1', name: 'Intel Core i7-14700K', price: 65000, specs: '20コア/28スレッド, P-core Max 5.5GHz', image: 'https://picsum.photos/400/300?random=3', affiliateUrl: 'https://affiliate-dummy-link.com/cpu1' },
            { id: 'cpu2', name: 'AMD Ryzen 7 7700X', price: 50000, specs: '8コア/16スレッド, Max 5.4GHz', image: 'https://picsum.photos/400/300?random=4', affiliateUrl: '' },
        ],
        // CPUクーラーカテゴリを追加
        cpucooler: [
            { id: 'cpucooler1', name: 'NZXT Kraken 360 RGB', price: 28000, specs: '360mm 水冷, LCDディスプレイ', image: 'https://picsum.photos/400/300?random=15', affiliateUrl: 'https://affiliate-dummy-link.com/cooler1' },
            { id: 'cpucooler2', name: 'Noctua NH-D15', price: 15000, specs: '空冷デュアルタワー, 高静音性', image: 'https://picsum.photos/400/300?random=16', affiliateUrl: 'https://affiliate-dummy-link.com/cooler2' },
            { id: 'cpucooler3', name: 'DeepCool AK620', price: 9000, specs: '空冷デュアルタワー', image: 'https://picsum.photos/400/300?random=17', affiliateUrl: '' },
        ],
        mb: [
            { id: 'mb1', name: 'ASUS PRIME Z790-A WIFI', price: 40000, specs: 'Intel Z790, ATX, DDR5', image: 'https://picsum.photos/400/300?random=5', affiliateUrl: 'https://affiliate-dummy-link.com/mb1' },
            { id: 'mb2', name: 'MSI MAG B650 TOMAHAWK', price: 30000, specs: 'AMD B650, ATX, DDR5', image: 'https://picsum.photos/400/300?random=6', affiliateUrl: '' },
        ],
        memory: [
            { id: 'mem1', name: 'DDR5-5600 32GB (16GBx2)', price: 18000, specs: 'CL36, デュアルチャネル', image: 'https://picsum.photos/400/300?random=7', affiliateUrl: 'https://affiliate-dummy-link.com/mem1' },
            { id: 'mem2', name: 'DDR4-3200 16GB (8GBx2)', price: 8000, specs: 'CL16, デュアルチャネル', image: 'https://picsum.photos/400/300?random=8', affiliateUrl: '' },
        ],
        vga: [
            { id: 'vga1', name: 'NVIDIA RTX 4070 Ti SUPER', price: 130000, specs: '16GB GDDR6X', image: 'https://picsum.photos/400/300?random=9', affiliateUrl: 'https://affiliate-dummy-link.com/vga1' },
            { id: 'vga2', name: 'AMD Radeon RX 7800 XT', price: 85000, specs: '16GB GDDR6', image: 'https://picsum.photos/400/300?random=10', affiliateUrl: '' },
        ],
        psu: [
            { id: 'psu1', name: 'Corsair RM1000e', price: 20000, specs: '1000W, 80 PLUS Gold', image: 'https://picsum.photos/400/300?random=11', affiliateUrl: 'https://affiliate-dummy-link.com/psu1' },
            { id: 'psu2', name: 'SilverStone Essential 750W', price: 12000, specs: '750W, 80 PLUS Bronze', image: 'https://picsum.photos/400/300?random=12', affiliateUrl: '' },
        ],
        ssd: [
            { id: 'ssd1', name: 'Crucial P5 Plus 2TB', price: 18000, specs: 'NVMe Gen4, 6600MB/s', image: 'https://picsum.photos/400/300?random=13', affiliateUrl: 'https://affiliate-dummy-link.com/ssd1' },
            { id: 'ssd2', name: 'Samsung 870 EVO 1TB', price: 12000, specs: 'SATA 3.0', image: 'https://picsum.photos/400/300?random=14', affiliateUrl: '' },
        ]
    };

    constructor() {
        this.componentData = {};
        this.selectedComponents = {};
        this.totalPrice = 0;
        this.savedConfigs = [];

        this.loadData();
        this.loadSavedConfigs();
        this.init();
    }

    /** アプリケーションの初期設定 */
    init() {
        this.renderCategories();
        this.setupListeners();
        this.updatePreview();
        this.updateTotalPrice();
        this.renderSavedConfigs();
    }

    /** 製品データをLocal Storageからロードまたはデフォルトを使用 */
    loadData() {
        try {
            const storedData = localStorage.getItem('pcBuilderComponentData');
            if (storedData) {
                this.componentData = JSON.parse(storedData);
            } else {
                this.componentData = this.DEFAULT_DATA;
                // 初回起動時にデフォルトデータを保存
                localStorage.setItem('pcBuilderComponentData', JSON.stringify(this.DEFAULT_DATA));
            }
        } catch (e) {
            console.error('製品データのロードに失敗しました。', e);
            this.componentData = this.DEFAULT_DATA;
        }
    }

    /** 保存された構成をLocal Storageからロード */
    loadSavedConfigs() {
        try {
            const storedConfigs = localStorage.getItem('pcBuilderSavedConfigs');
            if (storedConfigs) {
                this.savedConfigs = JSON.parse(storedConfigs);
            }
        } catch (e) {
            console.error('保存構成のロードに失敗しました。', e);
        }
    }

    /** 製品選択UIを動的にレンダリング */
    renderCategories() {
        const container = document.getElementById('component-categories');
        if (!container) return;

        container.innerHTML = '';
        
        // PREVIEW_ORDERに沿ってカテゴリをレンダリングすることで、表示順を制御
        this.PREVIEW_ORDER.forEach(categoryInfo => {
            const categoryKey = categoryInfo.key;
            const categoryItems = this.componentData[categoryKey];
            if (!categoryItems || categoryItems.length === 0) return; // データがないカテゴリはスキップ

            const categoryTitle = categoryInfo.name;
            const categoryIcon = categoryInfo.icon;

            const categorySection = document.createElement('div');
            categorySection.className = 'category-section';
            categorySection.dataset.category = categoryKey;
            
            // Header
            categorySection.innerHTML = `
                <div class="category-header">
                    <h3><i class="${categoryIcon}"></i> ${categoryTitle}</h3>
                    <span class="selected-item" data-category-status="${categoryKey}">未選択</span>
                </div>
                <div class="category-options">
                    ${categoryItems.map(item => `
                        <div class="option-card" data-component-id="${item.id}" data-category="${categoryKey}">
                            <div class="option-image">
                                ${item.image ? `<img src="${item.image}" alt="${item.name}">` : `<i class="${categoryIcon}"></i>`}
                            </div>
                            <div class="option-header">
                                <p class="option-name">${item.name}</p>
                                <p class="option-price">¥${item.price.toLocaleString()}</p>
                            </div>
                            <p class="option-specs">${item.specs}</p>
                            
                            ${item.affiliateUrl ? 
                                `<a href="${item.affiliateUrl}" target="_blank" class="btn btn-purchase" data-component-id="${item.id}" data-category="${categoryKey}">
                                    <i class="fas fa-shopping-cart"></i> 購入する
                                </a>` : 
                                `<button class="btn btn-select-only" data-component-id="${item.id}" data-category="${categoryKey}">
                                    <i class="fas fa-check"></i> 選択する
                                </button>`
                            }
                        </div>
                    `).join('')}
                </div>
            `;
            container.appendChild(categorySection);
        });

        // カードとボタンにイベントリスナーを設定
        this.setupCategoryEventListeners();
        
        // 既存の選択状態をハイライト
        this.restoreSelectionHighlights();
    }
    
    /** カテゴリカードのイベントリスナーを設定 */
    setupCategoryEventListeners() {
        // 「選択する」ボタンにリスナーを設定
        document.querySelectorAll('.btn-select-only').forEach(button => {
            const card = button.closest('.option-card');
            if (card) {
                button.addEventListener('click', (e) => {
                    e.stopPropagation(); // 親要素へのイベント伝播を停止
                    this.handleSelection(card);
                });
            }
        });
        
        // カードのボタン以外の領域がクリックされた場合にも選択できるように設定
        document.querySelectorAll('.option-card').forEach(card => {
            card.addEventListener('click', (e) => {
                // ボタン（購入/選択）がクリックされた場合は処理をスキップ
                if (e.target.closest('.btn-purchase') || e.target.closest('.btn-select-only')) {
                    return;
                }
                this.handleSelection(card);
            });
        });
    }


    /** 復元された構成のUIを更新 */
    restoreSelectionHighlights() {
        Object.keys(this.selectedComponents).forEach(category => {
            const selectedItem = this.selectedComponents[category];
            if (selectedItem) {
                 const card = document.querySelector(`.option-card[data-component-id="${selectedItem.id}"][data-category="${category}"]`);
                 if (card) {
                     card.classList.add('selected');
                 }
                 const statusEl = document.querySelector(`.selected-item[data-category-status="${category}"]`);
                 if(statusEl) statusEl.textContent = selectedItem.name;
            }
        });
    }

    /** コンポーネント選択時の処理 */
    handleSelection(card) {
        const category = card.dataset.category;
        const componentId = card.dataset.componentId;
        
        const selectedItem = this.componentData[category].find(item => item.id === componentId);
        if (!selectedItem) return;

        this.selectedComponents[category] = selectedItem;

        // UIの選択状態を更新 (ハイライト)
        document.querySelectorAll(`.option-card[data-category="${category}"]`).forEach(c => {
            c.classList.remove('selected');
        });
        card.classList.add('selected');

        // カテゴリヘッダーのステータスを更新
        document.querySelector(`.selected-item[data-category-status="${category}"]`).textContent = selectedItem.name;

        // プレビューと合計金額を更新
        this.updatePreview();
        this.updateTotalPrice();
    }

    /** プレビューパネルの表示を更新 */
    updatePreview() {
        // 1. PCケースのプレビューを更新
        const caseData = this.selectedComponents.case;
        const caseImageEl = document.querySelector('.case-actual-image');
        const caseIconEl = document.querySelector('.pc-case-preview .case-icon');
        const caseNameEl = document.getElementById('case-specs');

        if (caseData && caseImageEl && caseIconEl) {
            if (caseData.image) {
                caseImageEl.src = caseData.image;
                caseImageEl.style.display = 'block';
                caseIconEl.style.display = 'none';
            } else {
                caseImageEl.style.display = 'none';
                caseIconEl.style.display = 'block';
            }
            caseNameEl.innerHTML = `<p class="spec-value">${caseData.name}</p><p class="spec-value detail">${caseData.specs}</p>`;
        } else if (caseIconEl) {
            // ケースが未選択の場合
            caseImageEl.style.display = 'none';
            caseIconEl.style.display = 'block';
            caseNameEl.innerHTML = `<p class="spec-value detail">PCケースが未選択です</p>`;
        }

        // 2. その他のコンポーネントのプレビューを更新 (ご要望の順序通りにHTMLが構成されている前提)
        this.PREVIEW_ORDER.filter(c => c.key !== 'case').forEach(comp => {
            const data = this.selectedComponents[comp.key];
            const spanEl = document.querySelector(`.component-item.component-${comp.key} span[data-component="${comp.key}"]`);
            const itemEl = document.querySelector(`.component-item.component-${comp.key}`);
            
            if (spanEl) {
                if (data) {
                    spanEl.textContent = `${data.name} (¥${data.price.toLocaleString()})`;
                    itemEl.classList.add('selected');
                } else {
                    spanEl.textContent = '未選択';
                    itemEl.classList.remove('selected');
                }
            }
        });
    }

    /** 合計金額を計算し、表示を更新 */
    updateTotalPrice() {
        this.totalPrice = 0;
        
        Object.values(this.selectedComponents).forEach(item => {
            this.totalPrice += item.price;
        });

        const priceEl = document.getElementById('total-price');
        if (priceEl) {
            // アニメーション効果を適用
            priceEl.classList.add('updating');
            
            setTimeout(() => {
                priceEl.textContent = `¥ ${this.totalPrice.toLocaleString()}`;
                priceEl.classList.remove('updating');
            }, 300); // CSSのトランジション時間に合わせて調整
        }
    }
    
    /** すべてのイベントリスナーを設定 */
    setupListeners() {
        // メインアクションボタン
        document.getElementById('export-quote-btn').addEventListener('click', this.exportQuote.bind(this));
        document.getElementById('save-config-btn').addEventListener('click', this.saveConfig.bind(this));
        
        // 管理パネルボタン
        document.getElementById('export-data-btn').addEventListener('click', this.adminExportData.bind(this));
        document.getElementById('clear-data-btn').addEventListener('click', this.clearAllData.bind(this));
        document.getElementById('upload-area').addEventListener('click', () => document.getElementById('csv-file-input').click());
        document.getElementById('csv-file-input').addEventListener('change', this.adminImportCsv.bind(this));
        // ドラッグ&ドロップイベントのデフォルト動作を無効化
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            document.getElementById('upload-area').addEventListener(eventName, (e) => {
                e.preventDefault();
                e.stopPropagation();
                if (eventName === 'dragover') e.currentTarget.classList.add('dragover');
                if (eventName === 'dragleave' || eventName === 'drop') e.currentTarget.classList.remove('dragover');
            }, false);
        });
        document.getElementById('upload-area').addEventListener('drop', (e) => {
            const files = e.dataTransfer.files;
            if (files.length) {
                document.getElementById('csv-file-input').files = files;
                document.getElementById('csv-file-input').dispatchEvent(new Event('change'));
            }
        });
    }

    // --- 見積の出力 ---
    
    /** 現在の構成を見積としてCSV形式で出力（Excel互換） */
    exportQuote() {
        if (this.totalPrice === 0 || Object.keys(this.selectedComponents).length === 0) {
            alert('コンポーネントが選択されていません。');
            return;
        }

        const today = new Date().toLocaleDateString('ja-JP').replace(/\//g, '-');
        
        let csvContent = `\ufeff`; // UTF-8 BOM (文字化け対策)

        // --- Excelデザイン性を考慮したヘッダー部 ---
        csvContent += `"カスタムPC構成見積書",,,,\n`;
        csvContent += `"作成日","${today}",,,\n`;
        
        // ★ 修正箇所: 免責事項と価格の注意を追記
        csvContent += `"重要: 価格に関する免責事項",,,,\n`;
        csvContent += `"本ツールは見積作成を目的としており、表示価格はインポート時点のデータです。",,,\n`;
        csvContent += `"実際の販売価格は購入ボタン先のアマゾン（または他社）サイトで必ずご確認ください。弊社は価格変動に関する責任を一切負いません。 ",,,\n`;
        
        csvContent += `\n`; // 空行
        // 合計金額を目立たせる（右寄せ風）
        csvContent += `,,,"合計金額(税込)","¥ ${this.totalPrice.toLocaleString()}"\n`; 
        csvContent += `\n`; 
        // ----------------------------------------

        // コンポーネントリストのヘッダー
        csvContent += `"カテゴリ","製品名","スペック","単価(税込)","購入リンク"\n`;

        let subTotal = 0;

    // PREVIEW_ORDERの順番で見積書に出力
    this.PREVIEW_ORDER.forEach(comp => {
        const item = this.selectedComponents[comp.key];
        if (item) {
            subTotal += item.price;
            const categoryName = comp.name;
            const price = item.price.toLocaleString();
            
            // ★【修正箇所】HYPERLINK関数を生成し、短い表示テキストを実現
            let urlCellContent = '';
            if (item.affiliateUrl) {
                // 1. URL内の不要な文字列を徹底的にクリーンアップ
                // 過去の崩れの原因となった rel="nofollow"や、URL内の余分な二重引用符を削除
                const cleanUrl = item.affiliateUrl.replace(/" rel="nofollow"/g, '').replace(/\"/g, ''); 

                // 2. 表示テキストを決定（URLが「ショート」になったように見せるためのテキスト）
                const display_text = cleanUrl.includes('amazon.co.jp') ? 'Amazonで購入' : '購入リンク';
                
                // 3. ExcelのCSV読み込みで壊れないよう、関数全体を二重引用符で囲み、内部の引用符を二重引用符でエスケープする
                // 形式: ="=HYPERLINK(""URL"",""表示テキスト"")"
                urlCellContent = `"=HYPERLINK(""${cleanUrl}"",""${display_text}"")"`;
            } else {
                urlCellContent = '';
            }
            
            // すべてのフィールドをダブルクォーテーションで囲む
            // URLのセルは既に全体が引用符で囲まれた関数なので、そのまま埋め込む
            csvContent += `"${categoryName}","${item.name}","${item.specs || ''}","${price}",${urlCellContent}\n`;
        }
    });
// ...

        // --- Excelデザイン性を考慮したフッター部（小計として再掲） ---
        csvContent += `\n`;
        csvContent += `,,,"小計(税込)","¥ ${subTotal.toLocaleString()}"\n`;
        // ----------------------------------------------------

        // CSVファイルをダウンロード
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        if (link.download !== undefined) {
            const url = URL.createObjectURL(blob);
            link.setAttribute('href', url);
            link.setAttribute('download', `PC_Quote_${today}.csv`);
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            alert('見積書 (CSV形式) をエクスポートしました。');
        }
    }

    // --- 構成の保存と比較 ---
    
    /** 現在の構成を保存し、比較リストに表示 */
    saveConfig() {
        if (this.totalPrice === 0 || Object.keys(this.selectedComponents).length === 0) {
            alert('構成が空のため保存できません。');
            return;
        }

        const configName = `構成 ${this.savedConfigs.length + 1}`;
        const newConfig = {
            id: `config-${Date.now()}`,
            name: configName,
            totalPrice: this.totalPrice,
            timestamp: Date.now(),
            components: JSON.parse(JSON.stringify(this.selectedComponents))
        };

        this.savedConfigs.push(newConfig);
        localStorage.setItem('pcBuilderSavedConfigs', JSON.stringify(this.savedConfigs));
        this.renderSavedConfigs();
        alert(`${configName} を保存しました。`);
    }

    /** 保存された構成リストをレンダリング */
    renderSavedConfigs() {
        const container = document.getElementById('saved-config-list');
        const countEl = document.getElementById('saved-config-count');
        if (!container || !countEl) return;

        countEl.textContent = this.savedConfigs.length;
        container.innerHTML = '';

        this.savedConfigs.sort((a, b) => b.timestamp - a.timestamp);

        this.savedConfigs.forEach((config) => {
            const card = document.createElement('div');
            card.className = 'config-card';
            card.dataset.configId = config.id;
            
            // 構成の主要コンポーネント (CPUとVGA) を抜粋
            const cpuName = config.components.cpu ? config.components.cpu.name : 'CPU 未選択';
            const vgaName = config.components.vga ? config.components.vga.name : 'VGA 未選択';
            
            card.innerHTML = `
                <h4><i class="fas fa-save"></i> ${config.name}</h4>
                <p><strong>合計:</strong> ¥${config.totalPrice.toLocaleString()}</p>
                <p class="specs-summary"><i class="fas fa-microchip"></i> ${cpuName}</p>
                <p class="specs-summary"><i class="fas fa-temperature-arrow-up"></i> ${vgaName}</p>
                <button class="btn-clear-config" data-config-id="${config.id}"><i class="fas fa-times-circle"></i></button>
            `;

            card.addEventListener('click', (e) => {
                // 削除ボタン以外のクリックで復元
                if (!e.target.closest('.btn-clear-config')) {
                    this.loadConfig(config);
                }
            });
            
            // 削除ボタンリスナー
            card.querySelector('.btn-clear-config').addEventListener('click', (e) => {
                e.stopPropagation(); // カードの復元イベントを防止
                this.deleteConfig(config.id);
            });
            
            container.appendChild(card);
        });
    }

    /** 保存された構成をビルダーに復元（ロード） */
    loadConfig(config) {
        this.selectedComponents = {};
        
        // 選択状態を復元
        Object.keys(config.components).forEach(category => {
            this.selectedComponents[category] = config.components[category];
        });
        
        // UIを更新
        this.restoreSelectionHighlights();
        this.updatePreview();
        this.updateTotalPrice();
        alert(`「${config.name}」をビルダーに復元しました。`);
    }

    /** 構成を削除 */
    deleteConfig(configId) {
        if (!confirm('この構成を本当に削除しますか？')) return;
        
        this.savedConfigs = this.savedConfigs.filter(config => config.id !== configId);
        localStorage.setItem('pcBuilderSavedConfigs', JSON.stringify(this.savedConfigs));
        this.renderSavedConfigs();
    }


    // --- 管理機能 ---
    
    /** CSVファイルから製品データをインポート */
    adminImportCsv(event) {
        const file = event.target.files[0];
        if (!file) return;

        const reader = new FileReader();
        reader.onload = (e) => {
            try {
                const csvData = e.target.result;
                const newComponentData = this.parseCsv(csvData);
                
                if (Object.keys(newComponentData).length === 0) {
                    throw new Error('CSVファイルから有効なデータを抽出できませんでした。フォーマットを確認してください。');
                }

                this.componentData = newComponentData;
                localStorage.setItem('pcBuilderComponentData', JSON.stringify(this.componentData));
                
                // データクリア時に初期化されるため、UIの再レンダリングと初期化を実行
                this.selectedComponents = {};
                this.init(); 
                alert(`製品データを正常にインポートしました。総カテゴリ数: ${Object.keys(newComponentData).length}`);
            } catch (error) {
                console.error('CSVインポートエラー:', error);
                alert(`CSVインポートエラー: ${error.message}`);
            }
        };
        // UTF-8 BOM対応のため、テキストとして読み込み
        reader.readAsText(file, 'UTF-8');
    }

    /** CSVテキストを製品データオブジェクトにパース */
    parseCsv(csv) {
        const lines = csv.split('\n').filter(line => line.trim() !== '');
        if (lines.length < 2) return {};

        const result = {};
        // BOMを除去し、ヘッダーを読み取り
        const headers = lines[0].split(',').map(h => h.trim().toLowerCase().replace(/[\ufeff]/g, ''));
        
        // 期待されるヘッダーのインデックス (アフィリエイトURLを追加)
        const indices = {
            category: headers.indexOf('カテゴリ'),
            name: headers.indexOf('製品名'),
            price: headers.indexOf('価格'),
            specs: headers.indexOf('スペック'),
            image: headers.indexOf('画像url'),
            affiliateUrl: headers.indexOf('アフィリエイトurl') // ★追加項目
        };
        
        // 必須項目がない場合はエラー
        if (indices.category === -1 || indices.name === -1 || indices.price === -1 || indices.specs === -1) {
             throw new Error('CSVのヘッダーに「カテゴリ,製品名,価格,スペック」が不足しています。');
        }


        for (let i = 1; i < lines.length; i++) {
            const values = lines[i].split(/,(?=(?:(?:[^"]*"){2})*[^"]*$)/); // ダブルクォーテーション内のカンマを無視
            if (values.length < 4) continue; // 必須項目不足

            const item = {};
            const category = values[indices.category].toLowerCase().trim().replace(/^"|"$/g, '');
            if (!this.CATEGORIES.includes(category)) continue; // 定義されたカテゴリ以外はスキップ

            item.id = `${category}-${Date.now()}-${i}`; // 一意のIDを生成
            item.name = values[indices.name].trim().replace(/^"|"$/g, '');
            item.price = parseInt(values[indices.price].trim().replace(/[^0-9]/g, ''), 10);
            item.specs = values[indices.specs].trim().replace(/^"|"$/g, '');
            item.image = indices.image !== -1 ? values[indices.image].trim().replace(/^"|"$/g, '') : '';
            // アフィリエイトURLの値を取得
            item.affiliateUrl = indices.affiliateUrl !== -1 ? values[indices.affiliateUrl].trim().replace(/^"|"$/g, '') : '';

            
            if (!result[category]) {
                result[category] = [];
            }
            result[category].push(item);
        }
        return result;
    }

    /** 現在の製品データと構成データをクリア */
    clearAllData() {
        if (confirm('警告: すべての製品データ、選択中の構成、保存された構成を削除します。よろしいですか？')) {
            localStorage.removeItem('pcBuilderComponentData');
            localStorage.removeItem('pcBuilderSavedConfigs');
            this.componentData = this.DEFAULT_DATA;
            this.selectedComponents = {};
            this.savedConfigs = [];
            this.init(); // UIを再初期化
            alert('すべてのデータがクリアされ、デフォルト設定に戻りました。');
        }
    }

    /** 現在の製品データをCSV形式でエクスポート (管理者用) */
    adminExportData() {
        const today = new Date().toLocaleDateString('ja-JP').replace(/\//g, '-');
        let csvContent = `\ufeff`; // UTF-8 BOM

        // CSVヘッダーに画像URLとアフィリエイトURLを追加
        csvContent += `"カテゴリ","製品名","価格","スペック","画像URL","アフィリエイトURL"\n`;

        this.CATEGORIES.forEach(categoryKey => {
            const items = this.componentData[categoryKey];
            if (!items) return;

            items.forEach(item => {
                // 文字列にカンマが含まれる可能性があるため、ダブルクォーテーションで囲む
                csvContent += `"${categoryKey}","${item.name}","${item.price}","${item.specs || ''}","${item.image || ''}","${item.affiliateUrl || ''}"\n`;
            });
        });

        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        if (link.download !== undefined) {
            const url = URL.createObjectURL(blob);
            link.setAttribute('href', url);
            link.setAttribute('download', `PC_Product_Data_${today}.csv`);
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            alert('製品データ (CSV形式) をエクスポートしました。');
        }
    }
}

// ページロード時にアプリケーションを初期化
document.addEventListener('DOMContentLoaded', () => {
    new PCBuilder();
});